/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.snmp;

import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Consumer;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.snmp.SnmpClientBuilder;
import org.logstash.snmp.SnmpClientException;
import org.logstash.snmp.SnmpUtils;
import org.logstash.snmp.User;
import org.logstash.snmp.mib.MibManager;
import org.logstash.snmp.trap.SnmpTrapMessage;
import org.snmp4j.CommandResponder;
import org.snmp4j.CommandResponderEvent;
import org.snmp4j.CommunityTarget;
import org.snmp4j.MessageDispatcher;
import org.snmp4j.MessageDispatcherImpl;
import org.snmp4j.PDU;
import org.snmp4j.PDUv1;
import org.snmp4j.ScopedPDU;
import org.snmp4j.Session;
import org.snmp4j.Snmp;
import org.snmp4j.Target;
import org.snmp4j.TransportMapping;
import org.snmp4j.UserTarget;
import org.snmp4j.event.AuthenticationFailureEvent;
import org.snmp4j.event.AuthenticationFailureListener;
import org.snmp4j.event.ResponseEvent;
import org.snmp4j.log.Log4jLogFactory;
import org.snmp4j.log.LogFactory;
import org.snmp4j.mp.MPv1;
import org.snmp4j.mp.MPv2c;
import org.snmp4j.mp.MPv3;
import org.snmp4j.mp.MessageProcessingModel;
import org.snmp4j.mp.SnmpConstants;
import org.snmp4j.security.Priv3DES;
import org.snmp4j.security.PrivacyProtocol;
import org.snmp4j.security.SecurityLevel;
import org.snmp4j.security.SecurityModel;
import org.snmp4j.security.SecurityModels;
import org.snmp4j.security.SecurityProtocols;
import org.snmp4j.security.TSM;
import org.snmp4j.security.USM;
import org.snmp4j.smi.Address;
import org.snmp4j.smi.AssignableFromInteger;
import org.snmp4j.smi.AssignableFromLong;
import org.snmp4j.smi.GenericAddress;
import org.snmp4j.smi.OID;
import org.snmp4j.smi.OctetString;
import org.snmp4j.smi.TcpAddress;
import org.snmp4j.smi.TlsAddress;
import org.snmp4j.smi.UdpAddress;
import org.snmp4j.smi.Variable;
import org.snmp4j.smi.VariableBinding;
import org.snmp4j.transport.AbstractTransportMapping;
import org.snmp4j.transport.DefaultTcpTransportMapping;
import org.snmp4j.transport.DefaultUdpTransportMapping;
import org.snmp4j.transport.TLSTM;
import org.snmp4j.util.DefaultPDUFactory;
import org.snmp4j.util.MultiThreadedMessageDispatcher;
import org.snmp4j.util.PDUFactory;
import org.snmp4j.util.TableEvent;
import org.snmp4j.util.TableUtils;
import org.snmp4j.util.ThreadPool;
import org.snmp4j.util.TreeEvent;
import org.snmp4j.util.TreeUtils;
import org.snmp4j.util.WorkerPool;

public class SnmpClient
implements Closeable {
    private static final Logger logger = LogManager.getLogger(SnmpClient.class);
    private final MibManager mib;
    private final Snmp snmp;
    private final Set<String> supportedTransports;
    private final Set<Integer> supportedVersions;
    private final CountDownLatch stopCountDownLatch = new CountDownLatch(1);
    private Duration closeTimeoutDuration = Duration.ofMinutes(1L);
    private final String host;
    private final int port;
    private final boolean mapOidVariableValues;
    private final Map<OctetString, Integer> usmUsersSecurityLevel = new HashMap<OctetString, Integer>();

    public static SnmpClientBuilder builder(MibManager mib, Set<String> protocols) {
        return new SnmpClientBuilder(mib, protocols, 0);
    }

    public static SnmpClientBuilder builder(MibManager mib, Set<String> protocols, int port) {
        return new SnmpClientBuilder(mib, protocols, port);
    }

    SnmpClient(MibManager mib, Set<String> supportedTransports, Set<Integer> supportedVersions, String host, int port, String messageDispatcherPoolName, int messageDispatcherPoolSize, List<User> users, OctetString localEngineId, boolean mapOidVariableValues) throws IOException {
        this.mib = mib;
        this.host = host;
        this.port = port;
        this.supportedVersions = supportedVersions;
        this.supportedTransports = supportedTransports;
        this.mapOidVariableValues = mapOidVariableValues;
        users.forEach(p -> this.usmUsersSecurityLevel.put(p.getSecurityName(), p.getSecurityLevel()));
        SecurityProtocols.getInstance().addPredefinedProtocolSet(SecurityProtocols.SecurityProtocolSet.maxCompatibility);
        SecurityProtocols.getInstance().addPrivacyProtocol((PrivacyProtocol)new Priv3DES());
        if (supportedVersions.contains(3)) {
            SecurityModels.getInstance().addSecurityModel((SecurityModel)new TSM(localEngineId, false));
        }
        this.snmp = SnmpClient.createSnmpClient(supportedTransports, supportedVersions, host, port, localEngineId, users, messageDispatcherPoolName, messageDispatcherPoolSize);
    }

    private static Snmp createSnmpClient(Set<String> supportedTransports, Set<Integer> supportedVersions, String host, int port, OctetString localEngineId, List<User> users, String messageDispatcherPoolName, int messageDispatcherPoolSize) throws IOException {
        boolean engineBootCount = false;
        MessageDispatcher messageDispatcher = SnmpClient.createMessageDispatcher(localEngineId, supportedVersions, users, 0, messageDispatcherPoolName, messageDispatcherPoolSize);
        Snmp snmp = new Snmp(messageDispatcher);
        for (String transport : supportedTransports) {
            snmp.addTransportMapping(SnmpClient.createTransport(SnmpClient.parseAddress(transport, host, port)));
        }
        return snmp;
    }

    private static MessageDispatcher createMessageDispatcher(OctetString localEngineId, Set<Integer> supportedVersions, List<User> users, int engineBootCount, String messageDispatcherPoolName, int messageDispatcherPoolSize) {
        ThreadPool threadPool = ThreadPool.create((String)messageDispatcherPoolName, (int)messageDispatcherPoolSize);
        MessageDispatcherImpl dispatcherImpl = new MessageDispatcherImpl();
        MultiThreadedMessageDispatcher dispatcher = new MultiThreadedMessageDispatcher((WorkerPool)threadPool, (MessageDispatcher)dispatcherImpl);
        if (supportedVersions.contains(0)) {
            dispatcher.addMessageProcessingModel((MessageProcessingModel)new MPv1());
        }
        if (supportedVersions.contains(1)) {
            dispatcher.addMessageProcessingModel((MessageProcessingModel)new MPv2c());
        }
        if (supportedVersions.contains(3)) {
            MPv3 mpv3 = new MPv3(SnmpClient.createUsm(users, localEngineId, engineBootCount));
            mpv3.setCurrentMsgID(MPv3.randomMsgID((int)engineBootCount));
            dispatcher.addMessageProcessingModel((MessageProcessingModel)mpv3);
            if (logger.isDebugEnabled()) {
                dispatcherImpl.addAuthenticationFailureListener(new AuthenticationFailureListener(){

                    public <A extends Address> void authenticationFailure(AuthenticationFailureEvent<A> event) {
                        String message = SnmpConstants.usmErrorMessage((int)event.getError());
                        logger.debug("SNMP authentication failed. source: {}, reason: {} ({})", (Object)event.getAddress(), (Object)message, (Object)event.getError());
                    }
                });
            }
        }
        return dispatcher;
    }

    private static USM createUsm(List<User> users, OctetString localEngineID, int engineBootCount) {
        USM usm = new USM(SecurityProtocols.getInstance(), localEngineID, engineBootCount);
        if (users != null) {
            users.stream().map(User::getUsmUser).forEach(arg_0 -> ((USM)usm).addUser(arg_0));
        }
        return usm;
    }

    public void listen() throws IOException {
        this.getSnmp().listen();
    }

    public void trap(String[] allowedSecurityNames, Consumer<SnmpTrapMessage> consumer) throws IOException {
        this.doTrap(allowedSecurityNames, consumer, this.stopCountDownLatch);
    }

    void doTrap(String[] communities, final Consumer<SnmpTrapMessage> consumer, CountDownLatch stopCountDownLatch) throws IOException {
        final Set<String> allowedCommunities = Set.of(communities);
        this.getSnmp().addCommandResponder(new CommandResponder(){

            public <A extends Address> void processPdu(CommandResponderEvent<A> event) {
                logger.debug("SNMP Trap received: {}", event);
                int version = event.getMessageProcessingModel();
                String securityName = new String(event.getSecurityName());
                if (!SnmpClient.this.validateTrapMessage(version, securityName, event.getSecurityLevel(), allowedCommunities)) {
                    return;
                }
                Map<String, Object> trapEvent = SnmpClient.this.createTrapEvent(version, securityName, event.getPDU());
                HashMap<String, Object> formattedVarBindings = new HashMap<String, Object>(event.getPDU().getVariableBindings().size());
                for (VariableBinding binding : event.getPDU().getVariableBindings()) {
                    formattedVarBindings.put(SnmpClient.this.mib.map(binding.getOid()), SnmpClient.this.coerceVariable(binding.getVariable()));
                }
                SnmpTrapMessage trapMessage = new SnmpTrapMessage(version, event.getSecurityName(), event.getPeerAddress(), trapEvent, formattedVarBindings);
                consumer.accept(trapMessage);
            }
        });
        this.getSnmp().listen();
        if (logger.isInfoEnabled()) {
            String[] versions = (String[])this.supportedVersions.stream().map(SnmpUtils::parseSnmpVersion).toArray(String[]::new);
            logger.info("SNMP trap receiver started on host: {}, port: {}, transports: {}, versions: {}.", (Object)this.host, (Object)this.port, this.supportedTransports, (Object)versions);
        }
        try {
            stopCountDownLatch.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private boolean validateTrapMessage(int version, String securityName, int securityLevel, Set<String> allowedCommunities) {
        if (version < 3) {
            if (!allowedCommunities.isEmpty() && !allowedCommunities.contains(securityName)) {
                logger.debug("Received trap message with unknown community: '{}'. Skipping", (Object)securityName);
                return false;
            }
        } else {
            int userSecurityLevel = this.usmUsersSecurityLevel.get(OctetString.fromString((String)securityName));
            if (securityLevel < userSecurityLevel) {
                logger.debug("Unsupported security level {} by user {}, minimum security level is {}. Skipping", (Object)SecurityLevel.get((int)securityLevel), (Object)securityName, (Object)SecurityLevel.get((int)userSecurityLevel));
                return false;
            }
        }
        return true;
    }

    private Map<String, Object> createTrapEvent(int version, String securityName, PDU pdu) {
        HashMap<String, Object> trapEvent = new HashMap<String, Object>();
        trapEvent.put("version", SnmpUtils.parseSnmpVersion(version));
        trapEvent.put("type", PDU.getTypeString((int)pdu.getType()));
        if (version > 0) {
            int requestId = Objects.nonNull(pdu.getRequestID()) ? pdu.getRequestID().getValue() : 0;
            trapEvent.put("request_id", requestId);
            trapEvent.put("error_status", pdu.getErrorStatus());
            trapEvent.put("error_status_text", pdu.getErrorStatusText());
            trapEvent.put("error_index", pdu.getErrorIndex());
        } else if (pdu instanceof PDUv1) {
            PDUv1 pdUv1 = (PDUv1)pdu;
            trapEvent.put("enterprise", String.valueOf(pdUv1.getEnterprise()));
            trapEvent.put("agent_addr", String.valueOf(pdUv1.getAgentAddress()));
            trapEvent.put("generic_trap", pdUv1.getGenericTrap());
            trapEvent.put("specific_trap", pdUv1.getSpecificTrap());
            trapEvent.put("timestamp", pdUv1.getTimestamp());
        }
        if (version < 3) {
            trapEvent.put("community", securityName);
        }
        HashMap<String, Object> coercedVarBindings = new HashMap<String, Object>(pdu.getVariableBindings().size());
        for (VariableBinding binding : pdu.getVariableBindings()) {
            coercedVarBindings.put(binding.getOid().toString(), this.coerceVariable(binding.getVariable()));
        }
        trapEvent.put("variable_bindings", coercedVarBindings);
        return trapEvent;
    }

    public Map<String, Object> get(Target<Address> target, OID[] oids) throws IOException {
        this.validateTargetVersion(target);
        PDU pdu = this.createPDU(target, -96);
        pdu.addAll(VariableBinding.createFromOIDs((OID[])oids));
        ResponseEvent responseEvent = this.getSnmp().send(pdu, target);
        if (responseEvent == null) {
            return Collections.emptyMap();
        }
        Exception error = responseEvent.getError();
        if (error != null) {
            throw new SnmpClientException(String.format("error sending snmp get request to target %s: %s", target.getAddress(), error.getMessage()), error);
        }
        PDU responsePdu = responseEvent.getResponse();
        if (responsePdu == null) {
            throw new SnmpClientException(String.format("timeout sending snmp get request to target %s", target.getAddress()));
        }
        HashMap<String, Object> result = new HashMap<String, Object>();
        for (VariableBinding binding : responsePdu.getVariableBindings()) {
            String oid = this.mib.map(binding.getOid());
            result.put(oid, this.coerceVariable(binding.getVariable()));
        }
        return result;
    }

    public Map<String, Object> walk(Target<Address> target, OID oid) {
        this.validateTargetVersion(target);
        TreeUtils treeUtils = this.createGetTreeUtils();
        List events = treeUtils.getSubtree(target, oid);
        if (events == null || events.isEmpty()) {
            return Collections.emptyMap();
        }
        HashMap<String, Object> result = new HashMap<String, Object>();
        for (TreeEvent event : events) {
            if (event == null) continue;
            if (event.isError()) {
                throw new SnmpClientException(String.format("error sending snmp walk request to target %s: %s", target.getAddress(), event.getErrorMessage()), event.getException());
            }
            VariableBinding[] variableBindings = event.getVariableBindings();
            if (variableBindings == null) continue;
            for (VariableBinding variableBinding : variableBindings) {
                if (variableBinding == null) continue;
                result.put(this.mib.map(variableBinding.getOid()), this.coerceVariable(variableBinding.getVariable()));
            }
        }
        return result;
    }

    TreeUtils createGetTreeUtils() {
        return new TreeUtils((Session)this.getSnmp(), this.creatPDUFactory(-96));
    }

    public Map<String, List<Map<String, Object>>> table(Target<Address> target, String tableName, OID[] oids) {
        this.validateTargetVersion(target);
        TableUtils tableUtils = this.createGetTableUtils();
        List events = tableUtils.getTable(target, oids, null, null);
        if (events == null || events.isEmpty()) {
            return Collections.emptyMap();
        }
        ArrayList rows = new ArrayList(events.size());
        for (TableEvent event : events) {
            if (event == null) continue;
            if (event.isError()) {
                throw new SnmpClientException(String.format("error sending snmp table request to target %s: %s", target.getAddress(), event.getErrorMessage()), event.getException());
            }
            VariableBinding[] variableBindings = event.getColumns();
            if (variableBindings == null || variableBindings.length == 0) continue;
            HashMap<String, Object> row = new HashMap<String, Object>();
            row.put("index", String.valueOf(event.getIndex()));
            for (VariableBinding binding : variableBindings) {
                if (binding == null) continue;
                String mappedOid = this.mib.map(this.removeVariableOidIndex(binding.getOid(), event.getIndex()));
                Object value = this.coerceVariable(binding.getVariable());
                row.put(mappedOid, value);
            }
            rows.add(row);
        }
        return Collections.singletonMap(tableName, rows);
    }

    private OID removeVariableOidIndex(OID oid, OID eventIndex) {
        if (oid.rightMostCompare(eventIndex.size(), eventIndex) != 0) {
            return oid;
        }
        return oid.subOID(0, oid.size() - eventIndex.size());
    }

    TableUtils createGetTableUtils() {
        return new TableUtils((Session)this.getSnmp(), this.creatPDUFactory(-96));
    }

    Object coerceVariable(Variable variable) {
        if (variable.isException()) {
            switch (variable.getSyntax()) {
                case 129: {
                    return "error: no such instance currently exists at this OID";
                }
                case 128: {
                    return "error: no such object currently exists at this OID";
                }
                case 130: {
                    return "error: end of MIB view";
                }
            }
            return String.format("error: %s", variable.getSyntaxString());
        }
        if (variable.getSyntax() == 5) {
            return "null";
        }
        if (variable instanceof AssignableFromLong) {
            return variable.toLong();
        }
        if (variable instanceof AssignableFromInteger) {
            return variable.toInt();
        }
        if (this.mapOidVariableValues && variable instanceof OID) {
            return this.mib.map((OID)variable);
        }
        try {
            return variable.toString();
        }
        catch (Exception e) {
            String message = String.format("error: unable to read variable value. Syntax: %d (%s)", variable.getSyntax(), variable.getSyntaxString());
            logger.error(message);
            return message;
        }
    }

    private void validateTargetVersion(Target<Address> target) {
        if (!this.supportedVersions.contains(target.getVersion())) {
            throw new SnmpClientException(String.format("SNMP version `%s` is disabled", SnmpUtils.parseSnmpVersion(target.getVersion())));
        }
    }

    public Target<Address> createTarget(String address, String version, int retries, int timeout, String community, String securityName, String securityLevel) {
        Address targetAddress;
        CommunityTarget target;
        int snmpVersion = SnmpUtils.parseSnmpVersion(version);
        if (snmpVersion == 3) {
            Objects.requireNonNull(securityName, "security_name is required");
            Objects.requireNonNull(securityLevel, "security_level is required");
            target = new UserTarget();
            target.setSecurityLevel(SnmpUtils.parseSecurityLevel(securityLevel));
            target.setSecurityName(new OctetString(securityName));
            if (address.startsWith("tls")) {
                target.setSecurityModel(4);
            }
        } else {
            Objects.requireNonNull(community, "community is required");
            target = new CommunityTarget();
            target.setCommunity(new OctetString(community));
        }
        if ((targetAddress = GenericAddress.parse((String)address)) == null) {
            throw new IllegalArgumentException(String.format("Invalid or unknown host address: `%s`", address));
        }
        target.setAddress(targetAddress);
        target.setVersion(snmpVersion);
        target.setRetries(retries);
        target.setTimeout((long)timeout);
        return target;
    }

    private PDUFactory creatPDUFactory(int pduType) {
        return new DefaultPDUFactory(pduType);
    }

    private PDU createPDU(Target<Address> target, int pduType) {
        Object pdu = target.getVersion() == 3 ? new ScopedPDU() : (pduType == -92 ? new PDUv1() : new PDU());
        pdu.setType(pduType);
        return pdu;
    }

    private static Address parseAddress(String protocol, String host, int port) {
        String actualProtocol = Objects.nonNull(protocol) ? protocol.toLowerCase() : "udp";
        String actualHost = Objects.nonNull(host) ? host : "0.0.0.0";
        String address = String.format("%s/%d", actualHost, port);
        switch (actualProtocol) {
            case "udp": {
                return new UdpAddress(address);
            }
            case "tcp": {
                return new TcpAddress(address);
            }
            case "tls": {
                return new TlsAddress(address);
            }
        }
        throw new SnmpClientException(String.format("Invalid transport protocol specified '%s', expecting 'udp', 'tcp' or 'tls'", protocol));
    }

    private static AbstractTransportMapping<? extends Address> createTransport(Address address) throws IOException {
        if (address instanceof TlsAddress) {
            return new TLSTM((TlsAddress)address);
        }
        if (address instanceof TcpAddress) {
            return new DefaultTcpTransportMapping((TcpAddress)address);
        }
        return new DefaultUdpTransportMapping((UdpAddress)address);
    }

    SnmpClient setCloseTimeoutDuration(Duration closeTimeoutDuration) {
        this.closeTimeoutDuration = closeTimeoutDuration;
        return this;
    }

    @Override
    public void close() {
        try {
            CompletableFuture.runAsync(this::closeSnmpClient).get(this.closeTimeoutDuration.toMillis(), TimeUnit.MILLISECONDS);
        }
        catch (InterruptedException e) {
            logger.error("Current thread was interrupted while closing the SNMP client. Aborting", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        catch (TimeoutException e) {
            logger.error("Timed out while closing the SNMP client. Ignoring", (Throwable)e);
        }
        catch (ExecutionException e) {
            logger.error("Error closing the SNMP client. Ignoring", e.getCause());
        }
        catch (Exception e) {
            logger.error("Unexpected error closing the SNMP client. Ignoring", (Throwable)e);
        }
    }

    private void closeSnmpClient() {
        try {
            this.snmp.close();
        }
        catch (Exception e) {
            logger.error("Error closing SNMP client", (Throwable)e);
        }
        finally {
            this.stopCountDownLatch.countDown();
        }
    }

    final Snmp getSnmp() {
        return this.snmp;
    }

    public boolean isListening() {
        return this.getSnmp().getMessageDispatcher().getTransportMappings().stream().allMatch(TransportMapping::isListening);
    }

    static {
        LogFactory.setLogFactory((LogFactory)new Log4jLogFactory());
    }
}

