/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.elements.util;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.Socket;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.Principal;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Security;
import java.security.cert.CertPath;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509ExtendedKeyManager;
import javax.net.ssl.X509ExtendedTrustManager;
import javax.net.ssl.X509TrustManager;
import org.eclipse.californium.elements.util.Asn1DerDecoder;
import org.eclipse.californium.elements.util.CertPathUtil;
import org.eclipse.californium.elements.util.NotForAndroid;
import org.eclipse.californium.elements.util.PemReader;
import org.eclipse.californium.elements.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SslContextUtil {
    public static final Logger LOGGER;
    public static final String CLASSPATH_SCHEME = "classpath://";
    public static final String PARAMETER_SEPARATOR = "#";
    public static final String JKS_ENDING = ".jks";
    public static final String BKS_ENDING = ".bks";
    public static final String PKCS12_ENDING = ".p12";
    public static final String PEM_ENDING = ".pem";
    public static final String DEFAULT_ENDING = "*";
    public static final String JKS_TYPE = "JKS";
    public static final String BKS_TYPE = "BKS";
    public static final String PKCS12_TYPE = "PKCS12";
    public static final String PEM_TYPE = "PEM";
    public static final String DEFAULT_SSL_PROTOCOL = "TLSv1.2";
    private static final String SCHEME_DELIMITER = "://";
    private static final String DEFAULT_ALIAS = "californium";
    private static final Map<String, String> KEY_STORE_TYPES;
    private static final Map<String, KeyStoreConfiguration> KEY_STORE_CONFIGS;
    private static final Map<String, InputStreamFactory> INPUT_STREAM_FACTORIES;
    private static final KeyManager ANONYMOUS;
    private static final TrustManager TRUST_ALL;

    public static Certificate[] loadTrustedCertificates(String trust) throws IOException, GeneralSecurityException {
        if (null == trust) {
            throw new NullPointerException("trust must be provided!");
        }
        String[] parameters = trust.split(PARAMETER_SEPARATOR, 3);
        if (1 == parameters.length) {
            KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(parameters[0]);
            if (configuration.simpleStore != null) {
                return SslContextUtil.loadTrustedCertificates(parameters[0], null, null);
            }
        }
        if (3 != parameters.length) {
            throw new IllegalArgumentException("trust must comply the pattern <keystore#hexstorepwd#aliaspattern>");
        }
        return SslContextUtil.loadTrustedCertificates(parameters[0], parameters[2], StringUtil.hex2CharArray(parameters[1]));
    }

    public static Credentials loadCredentials(String credentials) throws IOException, GeneralSecurityException {
        if (null == credentials) {
            throw new NullPointerException("credentials must be provided!");
        }
        String[] parameters = credentials.split(PARAMETER_SEPARATOR, 4);
        if (1 == parameters.length) {
            KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(parameters[0]);
            if (configuration.simpleStore != null) {
                return SslContextUtil.loadCredentials(parameters[0], null, null, null);
            }
        }
        if (4 != parameters.length) {
            throw new IllegalArgumentException("credentials must comply the pattern <keystore#hexstorepwd#hexkeypwd#alias>");
        }
        return SslContextUtil.loadCredentials(parameters[0], parameters[3], StringUtil.hex2CharArray(parameters[1]), StringUtil.hex2CharArray(parameters[2]));
    }

    public static TrustManager[] loadTrustManager(String keyStoreUri, String aliasPattern, char[] storePassword) throws IOException, GeneralSecurityException {
        Certificate[] trustedCertificates = SslContextUtil.loadTrustedCertificates(keyStoreUri, aliasPattern, storePassword);
        return SslContextUtil.createTrustManager("trusts", trustedCertificates);
    }

    public static KeyManager[] loadKeyManager(String keyStoreUri, String alias, char[] storePassword, char[] keyPassword) throws IOException, GeneralSecurityException {
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.privateKey == null) {
                throw new IllegalArgumentException("credentials missing! No private key found!");
            }
            if (credentials.chain == null) {
                throw new IllegalArgumentException("credentials missing! No certificate chain found!");
            }
            return SslContextUtil.createKeyManager(alias, credentials.privateKey, credentials.chain);
        }
        if (null == keyPassword) {
            throw new NullPointerException("keyPassword must be provided!");
        }
        KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
        if (alias != null && !alias.isEmpty()) {
            KeyStore ksAlias = KeyStore.getInstance(ks.getType());
            ksAlias.load(null);
            KeyStore.Entry entry = ks.getEntry(alias, new KeyStore.PasswordProtection(keyPassword));
            if (null == entry) {
                throw new GeneralSecurityException("key stores '" + keyStoreUri + "' doesn't contain credentials for '" + alias + "'");
            }
            ksAlias.setEntry(alias, entry, new KeyStore.PasswordProtection(keyPassword));
            ks = ksAlias;
        }
        return SslContextUtil.createKeyManager(ks, keyPassword);
    }

    public static Certificate[] loadTrustedCertificates(String keyStoreUri, String aliasPattern, char[] storePassword) throws IOException, GeneralSecurityException {
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.trusts == null) {
                throw new IllegalArgumentException("no trusted x509 certificates found in '" + keyStoreUri + "'!");
            }
            return credentials.trusts;
        }
        KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
        Pattern pattern = null;
        if (null != aliasPattern && !aliasPattern.isEmpty()) {
            pattern = Pattern.compile(aliasPattern);
        }
        ArrayList<Certificate> trustedCertificates = new ArrayList<Certificate>();
        Enumeration<String> e = ks.aliases();
        while (e.hasMoreElements()) {
            Certificate certificate;
            Matcher matcher;
            String alias = e.nextElement();
            if (null != pattern && !(matcher = pattern.matcher(alias)).matches() || trustedCertificates.contains(certificate = ks.getCertificate(alias))) continue;
            trustedCertificates.add(certificate);
        }
        if (trustedCertificates.isEmpty()) {
            throw new IllegalArgumentException("no trusted x509 certificates found in '" + keyStoreUri + "' for '" + aliasPattern + "'!");
        }
        return trustedCertificates.toArray(new Certificate[trustedCertificates.size()]);
    }

    public static Credentials loadCredentials(String keyStoreUri, String alias, char[] storePassword, char[] keyPassword) throws IOException, GeneralSecurityException {
        KeyStore.Entry entry;
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.privateKey == null) {
                throw new IllegalArgumentException("credentials missing! No private key found!");
            }
            if (credentials.chain == null && credentials.publicKey == null) {
                throw new IllegalArgumentException("credentials missing! Neither certificate chain nor public key found!");
            }
            return credentials;
        }
        if (null == alias) {
            throw new NullPointerException("alias must be provided!");
        }
        if (alias.isEmpty()) {
            throw new IllegalArgumentException("alias must not be empty!");
        }
        if (null == keyPassword) {
            throw new NullPointerException("keyPassword must be provided!");
        }
        KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
        if (ks.entryInstanceOf(alias, KeyStore.PrivateKeyEntry.class) && (entry = ks.getEntry(alias, new KeyStore.PasswordProtection(keyPassword))) instanceof KeyStore.PrivateKeyEntry) {
            KeyStore.PrivateKeyEntry pkEntry = (KeyStore.PrivateKeyEntry)entry;
            Certificate[] chain = pkEntry.getCertificateChain();
            X509Certificate[] x509Chain = SslContextUtil.asX509Certificates(chain);
            return new Credentials(pkEntry.getPrivateKey(), null, x509Chain);
        }
        throw new IllegalArgumentException("no credentials found for '" + alias + "' in '" + keyStoreUri + "'!");
    }

    public static PrivateKey loadPrivateKey(String keyStoreUri, String alias, char[] storePassword, char[] keyPassword) throws IOException, GeneralSecurityException {
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.privateKey != null) {
                return credentials.privateKey;
            }
        } else {
            KeyStore.Entry entry;
            if (null == alias) {
                throw new NullPointerException("alias must be provided!");
            }
            if (alias.isEmpty()) {
                throw new IllegalArgumentException("alias must not be empty!");
            }
            if (null == keyPassword) {
                throw new NullPointerException("keyPassword must be provided!");
            }
            KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
            if (ks.entryInstanceOf(alias, KeyStore.PrivateKeyEntry.class) && (entry = ks.getEntry(alias, new KeyStore.PasswordProtection(keyPassword))) instanceof KeyStore.PrivateKeyEntry) {
                KeyStore.PrivateKeyEntry pkEntry = (KeyStore.PrivateKeyEntry)entry;
                return pkEntry.getPrivateKey();
            }
        }
        throw new IllegalArgumentException("no private key found for '" + alias + "' in '" + keyStoreUri + "'!");
    }

    public static PublicKey loadPublicKey(String keyStoreUri, String alias, char[] storePassword) throws IOException, GeneralSecurityException {
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.publicKey != null) {
                return credentials.publicKey;
            }
        } else {
            if (null == alias) {
                throw new NullPointerException("alias must be provided!");
            }
            if (alias.isEmpty()) {
                throw new IllegalArgumentException("alias must not be empty!");
            }
            KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
            Certificate[] chain = ks.getCertificateChain(alias);
            return chain[0].getPublicKey();
        }
        throw new IllegalArgumentException("no public key found for '" + alias + "' in '" + keyStoreUri + "'!");
    }

    public static X509Certificate[] loadCertificateChain(String keyStoreUri, String alias, char[] storePassword) throws IOException, GeneralSecurityException {
        KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(keyStoreUri);
        if (configuration.simpleStore != null) {
            Credentials credentials = SslContextUtil.loadSimpleKeyStore(keyStoreUri, configuration);
            if (credentials.chain == null) {
                throw new IllegalArgumentException("No certificate chain found!");
            }
            return credentials.chain;
        }
        if (null == alias) {
            throw new NullPointerException("alias must be provided!");
        }
        if (alias.isEmpty()) {
            throw new IllegalArgumentException("alias must not be empty!");
        }
        KeyStore ks = SslContextUtil.loadKeyStore(keyStoreUri, storePassword, configuration);
        Certificate[] chain = ks.getCertificateChain(alias);
        return SslContextUtil.asX509Certificates(chain);
    }

    public static void configureDefaults() {
        KEY_STORE_TYPES.clear();
        KEY_STORE_TYPES.put(JKS_ENDING, JKS_TYPE);
        KEY_STORE_TYPES.put(BKS_ENDING, BKS_TYPE);
        KEY_STORE_TYPES.put(PKCS12_ENDING, PKCS12_TYPE);
        KEY_STORE_TYPES.put(PEM_ENDING, PEM_TYPE);
        KEY_STORE_TYPES.put(DEFAULT_ENDING, KeyStore.getDefaultType());
        KEY_STORE_CONFIGS.put(JKS_TYPE, new KeyStoreConfiguration(JKS_TYPE, null));
        KEY_STORE_CONFIGS.put(BKS_TYPE, new KeyStoreConfiguration(BKS_TYPE, null));
        KEY_STORE_CONFIGS.put(PKCS12_TYPE, new KeyStoreConfiguration(PKCS12_TYPE, null));
        KEY_STORE_CONFIGS.put(PEM_TYPE, new KeyStoreConfiguration(PEM_TYPE, new SimpleKeyStore(){

            @Override
            public Credentials load(InputStream inputStream) throws GeneralSecurityException, IOException {
                return SslContextUtil.loadPemCredentials(inputStream);
            }
        }));
        INPUT_STREAM_FACTORIES.clear();
        INPUT_STREAM_FACTORIES.put(CLASSPATH_SCHEME, new ClassLoaderInputStreamFactory());
    }

    public static String configure(String ending, String keyStoreType) {
        if (ending == null) {
            throw new NullPointerException("ending must not be null!");
        }
        if (!ending.equals(DEFAULT_ENDING) && !ending.startsWith(".")) {
            throw new IllegalArgumentException("ending must start with \".\"!");
        }
        if (keyStoreType == null) {
            throw new NullPointerException("key store type must not be null!");
        }
        if (keyStoreType.isEmpty()) {
            throw new IllegalArgumentException("key store type must not be empty!");
        }
        return KEY_STORE_TYPES.put(ending.toLowerCase(), keyStoreType);
    }

    public static KeyStoreConfiguration configure(String keyStoreType, KeyStoreConfiguration config) {
        if (keyStoreType == null) {
            throw new NullPointerException("key store type must not be null!");
        }
        if (keyStoreType.isEmpty()) {
            throw new IllegalArgumentException("key store type must not be empty!");
        }
        if (config == null) {
            throw new NullPointerException("key store configuration must not be null!");
        }
        return KEY_STORE_CONFIGS.put(keyStoreType, config);
    }

    public static InputStreamFactory configure(String scheme, InputStreamFactory streamFactory) {
        if (scheme == null) {
            throw new NullPointerException("scheme must not be null!");
        }
        if (!scheme.endsWith(SCHEME_DELIMITER)) {
            throw new IllegalArgumentException("scheme must end with \"://\"!");
        }
        if (streamFactory == null) {
            throw new NullPointerException("stream factory must not be null!");
        }
        return INPUT_STREAM_FACTORIES.put(scheme.toLowerCase(), streamFactory);
    }

    private static KeyStoreConfiguration getKeyStoreConfigurationFromUri(String uri) throws GeneralSecurityException {
        int endingIndex;
        int lastPartIndex;
        String ending = null;
        String type = null;
        if (!uri.equals(DEFAULT_ENDING) && (lastPartIndex = uri.lastIndexOf(47)) < (endingIndex = uri.lastIndexOf(46))) {
            ending = uri.substring(endingIndex).toLowerCase();
            type = KEY_STORE_TYPES.get(ending);
        }
        if (type == null) {
            type = KEY_STORE_TYPES.get(DEFAULT_ENDING);
        }
        if (type == null) {
            throw new GeneralSecurityException("no key store type for " + uri);
        }
        KeyStoreConfiguration configuration = KEY_STORE_CONFIGS.get(type.toUpperCase());
        if (configuration == null) {
            throw new GeneralSecurityException("no key store configuration for " + type);
        }
        return configuration;
    }

    private static String getSchemeFromUri(String uri) {
        int schemeIndex = uri.indexOf(SCHEME_DELIMITER);
        if (0 < schemeIndex) {
            return uri.substring(0, schemeIndex + SCHEME_DELIMITER.length()).toLowerCase();
        }
        return null;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private static InputStream getInputStreamFromUri(String keyStoreUri) throws IOException {
        if (null == keyStoreUri) {
            throw new NullPointerException("keyStoreUri must be provided!");
        }
        InputStream inStream = null;
        String scheme = SslContextUtil.getSchemeFromUri(keyStoreUri);
        if (scheme == null) {
            String errorMessage = null;
            File file = new File(keyStoreUri);
            if (!file.exists()) {
                errorMessage = " doesn't exists!";
            } else if (!file.isFile()) {
                errorMessage = " is not a file!";
            } else if (!file.canRead()) {
                errorMessage = " could not be read!";
            }
            if (errorMessage != null) throw new IOException("URI: " + keyStoreUri + ", file: " + file.getAbsolutePath() + errorMessage);
            inStream = new FileInputStream(file);
        } else {
            InputStreamFactory streamFactory = INPUT_STREAM_FACTORIES.get(scheme);
            if (streamFactory != null) {
                inStream = streamFactory.create(keyStoreUri);
            }
        }
        if (inStream != null) return inStream;
        URL url = new URL(keyStoreUri);
        return url.openStream();
    }

    private static KeyStore loadKeyStore(String keyStoreUri, char[] storePassword, KeyStoreConfiguration configuration) throws GeneralSecurityException, IOException {
        if (null == storePassword) {
            throw new NullPointerException("storePassword must be provided!");
        }
        try (InputStream inStream = SslContextUtil.getInputStreamFromUri(keyStoreUri);){
            KeyStore keyStore = KeyStore.getInstance(configuration.type);
            keyStore.load(inStream, storePassword);
            KeyStore keyStore2 = keyStore;
            return keyStore2;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Credentials loadSimpleKeyStore(String keyStoreUri, KeyStoreConfiguration configuration) throws GeneralSecurityException, IOException {
        try (InputStream inputStream = SslContextUtil.getInputStreamFromUri(keyStoreUri);){
            Credentials credentials = configuration.simpleStore.load(inputStream);
            return credentials;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static Credentials loadPemCredentials(InputStream inputStream) throws GeneralSecurityException, IOException {
        try (PemReader reader = new PemReader(inputStream);){
            String tag;
            Asn1DerDecoder.Keys keys = new Asn1DerDecoder.Keys();
            ArrayList<Certificate> certificatesList = new ArrayList<Certificate>();
            CertificateFactory factory = CertificateFactory.getInstance("X.509");
            while ((tag = reader.readNextBegin()) != null) {
                Object read;
                byte[] decode = reader.readToEnd();
                if (decode == null) continue;
                if (tag.contains("CERTIFICATE")) {
                    certificatesList.add(factory.generateCertificate(new ByteArrayInputStream(decode)));
                    continue;
                }
                if (tag.contains("PRIVATE KEY")) {
                    read = Asn1DerDecoder.readPrivateKey(decode);
                    if (read == null) {
                        throw new GeneralSecurityException("private key type not supported!");
                    }
                    keys.add((Asn1DerDecoder.Keys)read);
                    continue;
                }
                if (tag.contains("PUBLIC KEY")) {
                    read = Asn1DerDecoder.readSubjectPublicKey(decode);
                    if (read == null) {
                        throw new GeneralSecurityException("public key type not supported!");
                    }
                    keys.setPublicKey((PublicKey)read);
                    continue;
                }
                LOGGER.warn("{} not supported!", (Object)tag);
            }
            if (keys.getPrivateKey() == null && keys.getPublicKey() == null) {
                ArrayList<Certificate> unique = new ArrayList<Certificate>();
                for (Certificate certificate : certificatesList) {
                    if (unique.contains(certificate)) continue;
                    unique.add(certificate);
                }
                Certificate[] certificates = unique.toArray(new Certificate[unique.size()]);
                Credentials credentials = new Credentials(certificates);
                return credentials;
            }
            CertPath certPath = factory.generateCertPath(certificatesList);
            List<? extends Certificate> path = certPath.getCertificates();
            X509Certificate[] x509CertificateArray = path.toArray(new X509Certificate[path.size()]);
            Credentials credentials = new Credentials(keys.getPrivateKey(), keys.getPublicKey(), x509CertificateArray);
            return credentials;
        }
    }

    public static X509Certificate[] asX509Certificates(Certificate[] certificates) {
        if (null == certificates || 0 == certificates.length) {
            throw new IllegalArgumentException("certificates missing!");
        }
        X509Certificate[] x509Certificates = new X509Certificate[certificates.length];
        for (int index = 0; certificates.length > index; ++index) {
            if (null == certificates[index]) {
                throw new IllegalArgumentException("[" + index + "] is null!");
            }
            try {
                x509Certificates[index] = (X509Certificate)certificates[index];
                continue;
            }
            catch (ClassCastException e) {
                throw new IllegalArgumentException("[" + index + "] is not a x509 certificate! Instead it's a " + certificates[index].getClass().getName());
            }
        }
        return x509Certificates;
    }

    public static void ensureUniqueCertificates(X509Certificate[] certificates) {
        HashSet<X509Certificate> set = new HashSet<X509Certificate>();
        for (X509Certificate certificate : certificates) {
            if (set.add(certificate)) continue;
            throw new IllegalArgumentException("Truststore contains certificates duplicates with subject: " + certificate.getSubjectX500Principal());
        }
    }

    public static SSLContext createSSLContext(String alias, PrivateKey privateKey, X509Certificate[] chain, Certificate[] trusts) throws GeneralSecurityException {
        return SslContextUtil.createSSLContext(alias, privateKey, chain, trusts, DEFAULT_SSL_PROTOCOL);
    }

    public static SSLContext createSSLContext(String alias, PrivateKey privateKey, X509Certificate[] chain, Certificate[] trusts, String protocol) throws GeneralSecurityException {
        if (null == alias) {
            alias = DEFAULT_ALIAS;
        }
        KeyManager[] keyManager = SslContextUtil.createKeyManager(alias, privateKey, chain);
        TrustManager[] trustManager = SslContextUtil.createTrustManager(alias, trusts);
        SSLContext sslContext = SSLContext.getInstance(protocol);
        sslContext.init(keyManager, trustManager, null);
        return sslContext;
    }

    public static KeyManager[] createKeyManager(String alias, PrivateKey privateKey, X509Certificate[] chain) throws GeneralSecurityException {
        if (null == privateKey) {
            throw new NullPointerException("private key must be provided!");
        }
        if (null == chain) {
            throw new NullPointerException("certificate chain must be provided!");
        }
        if (0 == chain.length) {
            throw new IllegalArgumentException("certificate chain must not be empty!");
        }
        if (null == alias) {
            alias = DEFAULT_ALIAS;
        }
        try {
            char[] key = "intern".toCharArray();
            KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(DEFAULT_ENDING);
            KeyStore ks = KeyStore.getInstance(configuration.type);
            ks.load(null);
            ks.setKeyEntry(alias, privateKey, key, chain);
            return SslContextUtil.createKeyManager(ks, key);
        }
        catch (IOException e) {
            throw new GeneralSecurityException(e.getMessage());
        }
    }

    public static TrustManager[] createTrustManager(String alias, Certificate[] trusts) throws GeneralSecurityException {
        if (null == trusts) {
            throw new NullPointerException("trusted certificates must be provided!");
        }
        if (0 == trusts.length) {
            throw new IllegalArgumentException("trusted certificates must not be empty!");
        }
        if (null == alias) {
            alias = DEFAULT_ALIAS;
        }
        try {
            int index = 1;
            KeyStoreConfiguration configuration = SslContextUtil.getKeyStoreConfigurationFromUri(DEFAULT_ENDING);
            KeyStore ks = KeyStore.getInstance(configuration.type);
            ks.load(null);
            for (Certificate certificate : trusts) {
                ks.setCertificateEntry(alias + index, certificate);
                ++index;
            }
            return SslContextUtil.createTrustManager(ks);
        }
        catch (IOException e) {
            throw new GeneralSecurityException(e.getMessage());
        }
    }

    public static KeyManager[] createAnonymousKeyManager() {
        return new KeyManager[]{ANONYMOUS};
    }

    @NotForAndroid
    public static TrustManager[] createTrustAllManager() {
        return new TrustManager[]{TRUST_ALL};
    }

    private static KeyManager[] createKeyManager(KeyStore store, char[] keyPassword) throws GeneralSecurityException {
        String algorithm = Security.getProperty("ssl.KeyManagerFactory.algorithm");
        KeyManagerFactory kmf = KeyManagerFactory.getInstance(algorithm);
        kmf.init(store, keyPassword);
        return kmf.getKeyManagers();
    }

    private static TrustManager[] createTrustManager(KeyStore store) throws GeneralSecurityException {
        String algorithm = Security.getProperty("ssl.TrustManagerFactory.algorithm");
        TrustManagerFactory tmf = TrustManagerFactory.getInstance(algorithm);
        tmf.init(store);
        return tmf.getTrustManagers();
    }

    static {
        X509TrustManager trustAll;
        LOGGER = LoggerFactory.getLogger(SslContextUtil.class);
        KEY_STORE_TYPES = new ConcurrentHashMap<String, String>();
        KEY_STORE_CONFIGS = new ConcurrentHashMap<String, KeyStoreConfiguration>();
        INPUT_STREAM_FACTORIES = new ConcurrentHashMap<String, InputStreamFactory>();
        ANONYMOUS = new AnonymousX509ExtendedKeyManager();
        Asn1DerDecoder.getEdDsaProvider();
        SslContextUtil.configureDefaults();
        try {
            trustAll = new X509ExtendedTrustAllManager();
        }
        catch (NoClassDefFoundError ex) {
            trustAll = new X509TrustAllManager();
        }
        TRUST_ALL = trustAll;
    }

    @NotForAndroid
    private static class X509ExtendedTrustAllManager
    extends X509ExtendedTrustManager {
        private X509ExtendedTrustAllManager() {
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            X509TrustAllManager.validateChain(chain, true);
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            X509TrustAllManager.validateChain(chain, false);
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return X509TrustAllManager.NO_ISSUERS;
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType, Socket socket) throws CertificateException {
            X509TrustAllManager.validateChain(chain, true);
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType, SSLEngine engine) throws CertificateException {
            X509TrustAllManager.validateChain(chain, true);
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType, Socket socket) throws CertificateException {
            X509TrustAllManager.validateChain(chain, false);
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType, SSLEngine engine) throws CertificateException {
            X509TrustAllManager.validateChain(chain, false);
        }
    }

    private static class X509TrustAllManager
    implements X509TrustManager {
        private static final X509Certificate[] NO_ISSUERS = new X509Certificate[0];

        private X509TrustAllManager() {
        }

        private static void validateChain(X509Certificate[] chain, boolean client) throws CertificateException {
            if (chain != null && chain.length > 0) {
                LOGGER.debug("check certificate {} for {}", (Object)chain[0].getSubjectDN(), (Object)(client ? "client" : "server"));
                if (!CertPathUtil.canBeUsedForAuthentication(chain[0], client)) {
                    LOGGER.debug("check certificate {} for {} failed on key-usage!", (Object)chain[0].getSubjectDN(), (Object)(client ? "client" : "server"));
                    throw new CertificateException("Key usage not proper for " + (client ? "client" : "server"));
                }
                LOGGER.trace("check certificate {} for {} succeeded on key-usage!", (Object)chain[0].getSubjectDN(), (Object)(client ? "client" : "server"));
                CertPath path = CertPathUtil.generateValidatableCertPath(Arrays.asList(chain), null);
                try {
                    CertPathUtil.validateCertificatePathWithIssuer(false, path, NO_ISSUERS);
                    LOGGER.trace("check certificate {}[{}] for {} validated!", new Object[]{chain[0].getSubjectDN(), chain.length, client ? "client" : "server"});
                }
                catch (GeneralSecurityException e) {
                    LOGGER.debug("check certificate {} for {} failed on {}!", new Object[]{chain[0].getSubjectDN(), client ? "client" : "server", e.getMessage()});
                    if (e instanceof CertificateException) {
                        throw (CertificateException)e;
                    }
                    throw new CertificateException(e);
                }
            }
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            X509TrustAllManager.validateChain(chain, true);
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            X509TrustAllManager.validateChain(chain, false);
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return NO_ISSUERS;
        }
    }

    private static class AnonymousX509ExtendedKeyManager
    extends X509ExtendedKeyManager {
        private AnonymousX509ExtendedKeyManager() {
        }

        @Override
        public String chooseEngineClientAlias(String[] keyType, Principal[] issuers, SSLEngine engine) {
            return null;
        }

        @Override
        public String chooseEngineServerAlias(String keyType, Principal[] issuers, SSLEngine engine) {
            return null;
        }

        @Override
        public String chooseClientAlias(String[] keyType, Principal[] issuers, Socket socket) {
            return null;
        }

        @Override
        public String chooseServerAlias(String keyType, Principal[] issuers, Socket socket) {
            return null;
        }

        @Override
        public X509Certificate[] getCertificateChain(String alias) {
            return null;
        }

        @Override
        public String[] getClientAliases(String keyType, Principal[] issuers) {
            return null;
        }

        @Override
        public PrivateKey getPrivateKey(String alias) {
            return null;
        }

        @Override
        public String[] getServerAliases(String keyType, Principal[] issuers) {
            return null;
        }
    }

    public static class KeyStoreConfiguration {
        public final String type;
        public final SimpleKeyStore simpleStore;

        public KeyStoreConfiguration(String type, SimpleKeyStore simpleStore) {
            this.type = type;
            this.simpleStore = simpleStore;
        }
    }

    public static interface SimpleKeyStore {
        public Credentials load(InputStream var1) throws GeneralSecurityException, IOException;
    }

    public static class Credentials {
        private final PrivateKey privateKey;
        private final PublicKey publicKey;
        private final X509Certificate[] chain;
        private final Certificate[] trusts;

        public Credentials(PrivateKey privateKey, PublicKey publicKey, X509Certificate[] chain) {
            if (chain != null) {
                if (chain.length == 0) {
                    chain = null;
                } else if (publicKey != null) {
                    if (!publicKey.equals(chain[0].getPublicKey())) {
                        throw new IllegalArgumentException("public key doesn't match certificate!");
                    }
                } else {
                    publicKey = chain[0].getPublicKey();
                }
            }
            this.privateKey = privateKey;
            this.chain = chain;
            this.publicKey = publicKey;
            this.trusts = null;
        }

        public Credentials(Certificate[] trusts) {
            this.privateKey = null;
            this.publicKey = null;
            this.chain = null;
            this.trusts = trusts;
        }

        public PrivateKey getPrivateKey() {
            return this.privateKey;
        }

        public PublicKey getPubicKey() {
            return this.publicKey;
        }

        public X509Certificate[] getCertificateChain() {
            return this.chain;
        }

        public Certificate[] getTrustedCertificates() {
            return this.trusts;
        }
    }

    private static class ClassLoaderInputStreamFactory
    implements InputStreamFactory {
        private ClassLoaderInputStreamFactory() {
        }

        @Override
        public InputStream create(String uri) throws IOException {
            String resource = uri.substring(SslContextUtil.CLASSPATH_SCHEME.length());
            InputStream inStream = Thread.currentThread().getContextClassLoader().getResourceAsStream(resource);
            if (null == inStream) {
                throw new IOException("'" + uri + "' not found!");
            }
            return inStream;
        }
    }

    public static interface InputStreamFactory {
        public InputStream create(String var1) throws IOException;
    }
}

